package com.androidbook.triviaquiz9;

import java.io.IOException;
import org.xmlpull.v1.XmlPullParserException;
import android.content.res.XmlResourceParser;
import android.os.Bundle;
import android.util.Log;
import android.widget.TabHost;
import android.widget.TableLayout;
import android.widget.TableRow;
import android.widget.TextView;
import android.widget.TabHost.TabSpec;

public class QuizScoresActivity extends QuizActivity {
    /** アクティビティが最初に生成されるときに呼び出される */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.scores);
        // タブの準備
        TabHost host = (TabHost) findViewById(R.id.TabHost1);
        host.setup();
        // 「全スコア」タブ
        TabSpec allScoresTab = host.newTabSpec("allTab");
        allScoresTab.setIndicator(getResources().getString(R.string.all_scores), getResources().getDrawable(
                android.R.drawable.star_on));
        allScoresTab.setContent(R.id.ScrollViewAllScores);
        host.addTab(allScoresTab);
        // 「フレンド」タブ
        TabSpec friendScoresTab = host.newTabSpec("friendsTab");
        friendScoresTab.setIndicator(getResources().getString(R.string.friends_scores), getResources().getDrawable(
                android.R.drawable.star_on));
        friendScoresTab.setContent(R.id.ScrollViewFriendScores);
        host.addTab(friendScoresTab);
        // デフォルトのタブの設定
        host.setCurrentTabByTag("allTab");
        // TableLayoutへの参照の取得
        TableLayout allScoresTable = (TableLayout) findViewById(R.id.TableLayout_AllScores);
        TableLayout friendScoresTable = (TableLayout) findViewById(R.id.TableLayout_FriendScores);
        // 各々のTableLayoutにカラム名の入った黄色いヘッダ行を追加
        initializeHeaderRow(allScoresTable);
        initializeHeaderRow(friendScoresTable);
        XmlResourceParser mockAllScores = getResources().getXml(R.xml.allscores);
        XmlResourceParser mockFriendScores = getResources().getXml(R.xml.friendscores);
        try {
            processScores(allScoresTable, mockAllScores);
            processScores(friendScoresTable, mockFriendScores);
        } catch (Exception e) {
            Log.e(DEBUG_TAG, "スコア読み込みエラー", e);
        }
    }

    /**
     * ヘッダ行として {@code TableRow} を {@code TableLayout} に（スタイル付きで）追加する
     * 
     * @param scoreTable
     *            ヘッダ行を追加する対象の {@code TableLayout}
     */
    private void initializeHeaderRow(TableLayout scoreTable) {
        // ヘッダ行となる TableRow の生成
        TableRow headerRow = new TableRow(this);
        int textColor = getResources().getColor(R.color.logo_color);
        float textSize = getResources().getDimension(R.dimen.help_text_size);
        addTextToRowWithValues(headerRow, getResources().getString(R.string.username), textColor, textSize);
        addTextToRowWithValues(headerRow, getResources().getString(R.string.score), textColor, textSize);
        addTextToRowWithValues(headerRow, getResources().getString(R.string.rank), textColor, textSize);
        scoreTable.addView(headerRow);
    }

    /**
     * XML形式のスコア情報を順次処理し、 {@code TableLayout} にデータを入れていく
     * 
     * @param scoreTable
     *            データを入れる対象の {@code TableLayout}
     * @param scores
     *            スコアデータを保持する標準の {@code XmlResourceParser}
     * @throws XmlPullParserException
     *            XMLエラーの際に発行される
     * @throws IOException
     *            XML読み込み中にIOエラーを起こした際に発行される
     */
    private void processScores(final TableLayout scoreTable, XmlResourceParser scores) throws XmlPullParserException,
            IOException {
        int eventType = -1;
        boolean bFoundScores = false;
        // XMLデータ中のスコアのレコードを見つける
        while (eventType != XmlResourceParser.END_DOCUMENT) {
            if (eventType == XmlResourceParser.START_TAG) {
                // タグ名を取得する(例　scoresまたはscore)
                String strName = scores.getName();
                if (strName.equals("score")) {
                    bFoundScores = true;
                    String scoreValue = scores.getAttributeValue(null, "score");
                    String scoreRank = scores.getAttributeValue(null, "rank");
                    String scoreUserName = scores.getAttributeValue(null, "username");
                    insertScoreRow(scoreTable, scoreValue, scoreRank, scoreUserName);
                }
            }
            eventType = scores.next();
        }
        // スコアが得られない場合の処理
        if (bFoundScores == false) {
            final TableRow newRow = new TableRow(this);
            TextView noResults = new TextView(this);
            noResults.setText(getResources().getString(R.string.no_scores));
            newRow.addView(noResults);
            scoreTable.addView(newRow);
        }
    }

    /**
     * {@code insertScoreRow()} ヘルパーメソッド -- {@code TableLayout} に
     * スコアを表示する新しい {@code TableRow} を追加する
     * 
     * @param scoreTable
     *            スコアを追加する対象の {@code TableLayout}
     * @param scoreValue
     *            得点
     * @param scoreRank
     *            ランク
     * @param scoreUserName
     *            ユーザー名
     */
    private void insertScoreRow(final TableLayout scoreTable, String scoreValue, String scoreRank, String scoreUserName) {
        final TableRow newRow = new TableRow(this);
        int textColor = getResources().getColor(R.color.title_color);
        float textSize = getResources().getDimension(R.dimen.help_text_size);
        addTextToRowWithValues(newRow, scoreUserName, textColor, textSize);
        addTextToRowWithValues(newRow, scoreValue, textColor, textSize);
        addTextToRowWithValues(newRow, scoreRank, textColor, textSize);
        scoreTable.addView(newRow);
    }

    /**
     * {@code addTextToRowWithValues()} ヘルパーメソッド -- {@code TableRow} に
     * 3列の {@code TextView} データを（スタイル付きで）追加する
     * 
     * @param tableRow
     *            文字列を追加する対象の {@code TableRow}
     * @param text
     *            追加する文字列
     * @param textColor
     *            文字列のカラー
     * @param textSize
     *            文字列のフォントサイズ
     */
    private void addTextToRowWithValues(final TableRow tableRow, String text, int textColor, float textSize) {
        TextView textView = new TextView(this);
        textView.setTextSize(textSize);
        textView.setTextColor(textColor);
        textView.setText(text);
        tableRow.addView(textView);
    }
}
