package com.androidbook.triviaquiz15_2;

import java.io.IOException;
import java.net.URL;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.res.XmlResourceParser;
import android.os.AsyncTask;
import android.os.Bundle;
import android.util.Log;
import android.view.Window;
import android.widget.TabHost;
import android.widget.TableLayout;
import android.widget.TableRow;
import android.widget.TextView;
import android.widget.TabHost.TabSpec;

public class QuizScoresActivity extends QuizActivity {

    int mProgressCounter = 0;
    ScoreDownloaderTask allScoresDownloader;
    ScoreDownloaderTask friendScoresDownloader;

    /** アクティビティが最初に生成されるときに呼び出される */

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        // 第15章で次の行を追加
        requestWindowFeature(Window.FEATURE_INDETERMINATE_PROGRESS); // 経過非表示のプログレスバーインジケータをタイトルバーに表示する前にこの行を追加しておかないと、正しく表示されない
        setContentView(R.layout.scores);
        // タブの準備
        TabHost host = (TabHost) findViewById(R.id.TabHost1);
        host.setup();
        // 「全スコア」タブ
        TabSpec allScoresTab = host.newTabSpec("allTab");
        allScoresTab.setIndicator(getResources().getString(R.string.all_scores),
                                  getResources().getDrawable(android.R.drawable.star_on));
        allScoresTab.setContent(R.id.ScrollViewAllScores);
        host.addTab(allScoresTab);
        // 「フレンド」タブ
        TabSpec friendScoresTab = host.newTabSpec("friendsTab");
        friendScoresTab.setIndicator(getResources().getString(R.string.friends_scores),
                                     getResources().getDrawable(android.R.drawable.star_on));
        friendScoresTab.setContent(R.id.ScrollViewFriendScores);
        host.addTab(friendScoresTab);
        // デフォルトのタブの設定
        host.setCurrentTabByTag("allTab");
        // TableLayoutへの参照の取得
        TableLayout allScoresTable = (TableLayout) findViewById(R.id.TableLayout_AllScores);
        TableLayout friendScoresTable = (TableLayout) findViewById(R.id.TableLayout_FriendScores);
        // 各々のTableLayoutにカラム名の入った黄色いヘッダ行を追加
        initializeHeaderRow(allScoresTable);
        initializeHeaderRow(friendScoresTable);
        
        // 第15章で変更　<---ここから
        allScoresDownloader = new ScoreDownloaderTask();
        allScoresDownloader.execute(TRIVIA_SERVER_SCORES, allScoresTable);
        
        SharedPreferences prefs = getSharedPreferences(GAME_PREFERENCES, Context.MODE_PRIVATE);
        Integer playerId = prefs.getInt(GAME_PREFERENCES_PLAYER_ID, -1);
        
        if (playerId != -1) {
            friendScoresDownloader = new ScoreDownloaderTask();
            friendScoresDownloader.execute(TRIVIA_SERVER_SCORES + "?playerId=" + playerId, friendScoresTable);
        }
        // 第15章で変更　ここまで--->
    }

    @Override
    protected void onPause() {
        if (allScoresDownloader != null && allScoresDownloader.getStatus() != AsyncTask.Status.FINISHED) {
            allScoresDownloader.cancel(true);
        }
        if (friendScoresDownloader != null && friendScoresDownloader.getStatus() != AsyncTask.Status.FINISHED) {
            friendScoresDownloader.cancel(true);
        }
        super.onPause();
    }

    /**
     * ヘッダ行として {@code TableRow} を {@code TableLayout} に（スタイル付きで）追加する
     * 
     * @param scoreTable
     *            ヘッダ行を追加する対象の {@code TableLayout}
     */
    private void initializeHeaderRow(TableLayout scoreTable) {
        // ヘッダ行となる TableRow の生成
        TableRow headerRow = new TableRow(this);

        int textColor = getResources().getColor(R.color.logo_color);
        float textSize = getResources().getDimension(R.dimen.help_text_size);

        addTextToRowWithValues(headerRow, getResources().getString(R.string.username), textColor, textSize);
        addTextToRowWithValues(headerRow, getResources().getString(R.string.score), textColor, textSize);
        addTextToRowWithValues(headerRow, getResources().getString(R.string.rank), textColor, textSize);
        scoreTable.addView(headerRow);
    }

    /**
     * {@code addTextToRowWithValues()} ヘルパーメソッド -- {@code TableRow} に
     * 3列の {@code TextView} データを（スタイル付きで）追加する
     * 
     * @param tableRow
     *            文字列を追加する対象の {@code TableRow}
     * @param text
     *            追加する文字列
     * @param textColor
     *            文字列のカラー
     * @param textSize
     *            文字列のフォントサイズ
     */
    private void addTextToRowWithValues(final TableRow tableRow, String text, int textColor, float textSize) {
        TextView textView = new TextView(this);
        textView.setTextSize(textSize);
        textView.setTextColor(textColor);
        textView.setText(text);
        tableRow.addView(textView);
    }

    // ScoreDownloaderTask 第15章で追加し、ヘルパーメソッドをこの中に移動
    
    private class ScoreDownloaderTask extends AsyncTask<Object, String, Boolean> {
        private static final String DEBUG_TAG = "ScoreDownloaderTask";
        TableLayout table;

        @Override
        protected void onCancelled() {
            Log.i(DEBUG_TAG, "onCancelled");
            mProgressCounter--;
            if (mProgressCounter <= 0) {
                mProgressCounter = 0;
                QuizScoresActivity.this.setProgressBarIndeterminateVisibility(false);
            }
        }

        @Override
        protected void onPostExecute(Boolean result) {
            Log.i(DEBUG_TAG, "onPostExecute");
            mProgressCounter--;
            if (mProgressCounter <= 0) {
                mProgressCounter = 0;
                QuizScoresActivity.this.setProgressBarIndeterminateVisibility(false);
            }
        }

        @Override
        protected void onPreExecute() {
            mProgressCounter++;
            QuizScoresActivity.this.setProgressBarIndeterminateVisibility(true);
        }

        @Override
        protected void onProgressUpdate(String... values) {

            if (values.length == 3) {
                String scoreValue = values[0];
                String scoreRank = values[1];
                String scoreUserName = values[2];
                insertScoreRow(table, scoreValue, scoreRank, scoreUserName);
            } else {
                final TableRow newRow = new TableRow(QuizScoresActivity.this);
                TextView noResults = new TextView(QuizScoresActivity.this);
                noResults.setText(getResources().getString(R.string.no_scores));
                newRow.addView(noResults);
                table.addView(newRow);
            }

        }

        @Override
        protected Boolean doInBackground(Object... params) {
            boolean result = false;
            String pathToScores = (String) params[0];
            table = (TableLayout) params[1];

            XmlPullParser scores;
            try {
                URL xmlUrl = new URL(pathToScores);
                scores = XmlPullParserFactory.newInstance().newPullParser();
                scores.setInput(xmlUrl.openStream(), null);
            } catch (XmlPullParserException e) {
                scores = null;
            } catch (IOException e) {
                scores = null;
            }

            if (scores != null) {
                try {
                    processScores(scores);
                } catch (XmlPullParserException e) {
                    Log.e(DEBUG_TAG, "XmlPullParserが処理に失敗しました", e);
                } catch (IOException e) {
                    Log.e(DEBUG_TAG, "XMLのパース中にIOエラーが発生しました", e);
                }
            }

            return result;
        }

        /**
         * {@code insertScoreRow()} ヘルパーメソッド -- {@code TableLayout} に
         * スコアを表示する新しい {@code TableRow} を追加する
         * 
         * @param scoreTable
         *            スコアを追加する対象の {@code TableLayout}
         * @param scoreValue
         *            得点
         * @param scoreRank
         *            ランク
         * @param scoreUserName
         *            ユーザー名
         */
        private void insertScoreRow(final TableLayout scoreTable, String scoreValue, String scoreRank, String scoreUserName) {
            final TableRow newRow = new TableRow(QuizScoresActivity.this);  	// 第15章でここも変更
            int textColor = getResources().getColor(R.color.title_color);
            float textSize = getResources().getDimension(R.dimen.help_text_size);
            addTextToRowWithValues(newRow, scoreUserName, textColor, textSize);
            addTextToRowWithValues(newRow, scoreValue, textColor, textSize);
            addTextToRowWithValues(newRow, scoreRank, textColor, textSize);
            scoreTable.addView(newRow);
        }

        /**
         * XML形式のスコア情報を順次処理し、 {@code TableLayout} にデータを入れていく
         * 
         * @param scores
         *            スコアデータを保持する標準の {@code XmlResourceParser}
         * @throws XmlPullParserException
         *            XMLエラーの際に発行される
         * @throws IOException
         *            XML読み込み中にIOエラーを起こした際に発行される
         */
        private void processScores(XmlPullParser scores) throws XmlPullParserException, IOException {
            int eventType = -1;
            boolean bFoundScores = false;

            // XMLの中から得点情報を探し出す
            while (eventType != XmlResourceParser.END_DOCUMENT) {
                if (eventType == XmlResourceParser.START_TAG) {

                    // タグ名（例：scores、score）を取得する
                    String strName = scores.getName();

                    if (strName.equals("score")) {
                        bFoundScores = true;
                        String scoreValue = scores.getAttributeValue(null, "score");
                        String scoreRank = scores.getAttributeValue(null, "rank");
                        String scoreUserName = scores.getAttributeValue(null, "username");
                        publishProgress(scoreValue, scoreRank, scoreUserName);
                    }
                }
                eventType = scores.next();
            }

            // 得点情報を得られなかった場合の処理
            if (bFoundScores == false) {
                publishProgress();
            }
        }

    }

}
