package com.androidbook.triviaquiz15_2;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Calendar;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.BasicResponseHandler;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.DatePickerDialog;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.DialogInterface.OnCancelListener;
import android.content.SharedPreferences.Editor;
import android.graphics.Bitmap;
import android.graphics.Bitmap.CompressFormat;
import android.location.Address;
import android.location.Geocoder;
import android.location.Location;
import android.location.LocationManager;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.provider.MediaStore.Images.Media;
import android.telephony.TelephonyManager;
import android.text.Editable;
import android.text.TextWatcher;
import android.text.format.DateFormat;
import android.text.format.Time;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.DatePicker;
import android.widget.EditText;
import android.widget.ImageButton;
import android.widget.Spinner;
import android.widget.TextView;

public class QuizSettingsActivity extends QuizActivity {
    
    SharedPreferences mGameSettings;
    GPSCoords mFavPlaceCoords;
    
    static final int DATE_DIALOG_ID = 0;
    static final int PASSWORD_DIALOG_ID = 1;
    static final int PLACE_DIALOG_ID = 2;
    
    static final int TAKE_AVATAR_CAMERA_REQUEST = 1;
    static final int TAKE_AVATAR_GALLERY_REQUEST = 2;
    
    /** アクティビティが最初に生成されるときに呼び出される */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.settings);
        
        // 共有プレファレンスの読み込み
        mGameSettings = getSharedPreferences(GAME_PREFERENCES, Context.MODE_PRIVATE);
        
        // アバター画像を表示／設定するボタンの初期化
        initAvatar();
        // ニックネーム入力フィールドの初期化
        initNicknameEntry();
        // メールアドレス入力フィールドの初期化
        initEmailEntry();
        // パスワード入力ダイアログの初期化
        initPasswordChooser();
        // 生年月日入力用日付ピッカーの初期化
        initDatePicker();
        // 性別を選択するスピナーの初期化
        initGenderSpinner();
        // 「お気に入りの場所」選択画面の初期化
        initFavoritePlacePicker();
    }
    
    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        
        switch (requestCode) {
            case TAKE_AVATAR_CAMERA_REQUEST:
                
                if (resultCode == Activity.RESULT_CANCELED) {
                    // アバターを撮影するカメラモードのキャンセル
                } else if (resultCode == Activity.RESULT_OK) {
                    
                    // 写真撮影完了。デフォルトで得られる縮小版の撮影画像を使用
                    Bitmap cameraPic = (Bitmap) data.getExtras().get("data");
                    if (cameraPic != null) {
                        try {
                            saveAvatar(cameraPic);
                        } catch (Exception e) {
                            Log.e(DEBUG_TAG, "カメラ画像をパラメータにsaveAvatar()を呼び出したところ失敗しました。", e);
                        }
                    }
                }
                break;
            case TAKE_AVATAR_GALLERY_REQUEST:
                
                if (resultCode == Activity.RESULT_CANCELED) {
                    // アバターを選択する画像ギャラリーのリクエストのキャンセル
                } else if (resultCode == Activity.RESULT_OK) {
                    
                    // 選択された画像の取得
                    Uri photoUri = data.getData();
                    if (photoUri != null) {
                        try {
                            int maxLength = 75;
                            // フルサイズの画像では大き過ぎる。アバターとしてふさわしいサイズに縮小
                            Bitmap galleryPic = Media.getBitmap(getContentResolver(), photoUri);
                            Bitmap scaledGalleryPic = createScaledBitmapKeepingAspectRatio(galleryPic, maxLength);
                            saveAvatar(scaledGalleryPic);
                        } catch (Exception e) {
                            Log.e(DEBUG_TAG, "ギャラリーで選択した画像をパラメータにsaveAvatar()を呼び出したところ失敗しました。", e);
                        }
                    }
                }
                break;
        }
    }
    
    /**
     * アスペクト比（縦横比）を維持したままBitmapを拡大縮小する
     * 
     * @param bitmap
     *            拡大縮小の対象となる{@code Bitmap}
     * @param maxSide
     *            縦幅もしくは横幅の最大値
     * @return 新しく作成された拡大／縮小後の{@code Bitmap}
     */
    private Bitmap createScaledBitmapKeepingAspectRatio(Bitmap bitmap, int maxSide) {
        int orgHeight = bitmap.getHeight();
        int orgWidth = bitmap.getWidth();
        
        // 長いほうの辺が75pxを越えないように拡大／縮小
        int scaledWidth = (orgWidth >= orgHeight) ? maxSide : (int) ((float) maxSide * ((float) orgWidth / (float) orgHeight));
        int scaledHeight = (orgHeight >= orgWidth) ? maxSide : (int) ((float) maxSide * ((float) orgHeight / (float) orgWidth));
        
        // 拡大／縮小後のBitmapを生成
        Bitmap scaledGalleryPic = Bitmap.createScaledBitmap(bitmap, scaledWidth, scaledHeight, true);
        return scaledGalleryPic;
    }
    
    /**
     * Bitmapオブジェクトをavatar.jpgとして保存
     * 
     * @param avatar
     *            ファイルに保存するBitmapオブジェクト
     */
    private void saveAvatar(Bitmap avatar) {
        String strAvatarFilename = "avatar.jpg";
        try {
            avatar.compress(CompressFormat.JPEG, 100, openFileOutput(strAvatarFilename, MODE_PRIVATE));
        } catch (Exception e) {
            Log.e(DEBUG_TAG, "アバター画像の圧縮と保存に失敗しました。", e);
        }
        
        Uri imageUriToSaveCameraImageTo = Uri.fromFile(new File(QuizSettingsActivity.this.getFilesDir(), strAvatarFilename));
        
        Editor editor = mGameSettings.edit();
        editor.putString(GAME_PREFERENCES_AVATAR, imageUriToSaveCameraImageTo.getPath());
        editor.commit();
        
        // 「設定」画面を更新
        ImageButton avatarButton = (ImageButton) findViewById(R.id.ImageButton_Avatar);
        String strAvatarUri = mGameSettings.getString(GAME_PREFERENCES_AVATAR, "android.resource://com.androidbook.triviaquiz15_2/drawable/avatar");
        Uri imageUri = Uri.parse(strAvatarUri);
        avatarButton.setImageURI(null); // ImageButtonを更新する際のウラ技。ImageButtonは表示していた画像をキャッシュしようとするが、nullを渡すとリセットされ、キャッシュがクリアされる。
        avatarButton.setImageURI(imageUri);
    }
    
    @Override
    protected void onDestroy() {
        
        Log.d(DEBUG_TAG, "SHARED PREFERENCES");
        Log.d(DEBUG_TAG, "ニックネーム: " + mGameSettings.getString(GAME_PREFERENCES_NICKNAME, "未設定"));
        Log.d(DEBUG_TAG, "メールアドレス: " + mGameSettings.getString(GAME_PREFERENCES_EMAIL, "未設定"));
        Log.d(DEBUG_TAG, "性別（男=1, 女=2, 秘密=0）: " + mGameSettings.getInt(GAME_PREFERENCES_GENDER, 0));
        Log.d(DEBUG_TAG, "パスワード: " + mGameSettings.getString(GAME_PREFERENCES_PASSWORD, "未設定"));
        Log.d(DEBUG_TAG, "生年月日: "
              + DateFormat.format("yyyy年 MMMM dd日", mGameSettings.getLong(GAME_PREFERENCES_DOB, 0)));
        Log.d(DEBUG_TAG, "アバター: " + mGameSettings.getString(GAME_PREFERENCES_AVATAR, "未設定"));
        Log.d(DEBUG_TAG, "お気に入りの場所名: " + mGameSettings.getString(GAME_PREFERENCES_FAV_PLACE_NAME, "未設定"));
        Log.d(DEBUG_TAG, "お気に入りの場所の緯度: " + mGameSettings.getFloat(GAME_PREFERENCES_FAV_PLACE_LAT, 0));
        Log.d(DEBUG_TAG, "お気に入りの場所の経度: " + mGameSettings.getFloat(GAME_PREFERENCES_FAV_PLACE_LONG, 0));
        
        super.onDestroy();
    }
    
    /**
     * アバター画像を表示／設定するボタンの初期化
     */
    private void initAvatar() {
        // アバターボタンの初期化
        ImageButton avatarButton = (ImageButton) findViewById(R.id.ImageButton_Avatar);
        
        if (mGameSettings.contains(GAME_PREFERENCES_AVATAR)) {
            String strAvatarUri = mGameSettings.getString(GAME_PREFERENCES_AVATAR, "android.resource://com.androidbook.triviaquiz15_2/drawable/avatar");
            Uri imageUri = Uri.parse(strAvatarUri);
            avatarButton.setImageURI(imageUri);
        } else {
            avatarButton.setImageResource(R.drawable.avatar);
        }
        
        avatarButton.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                String strAvatarPrompt = "アバターに使う写真を撮影してください。";
                Intent pictureIntent = new Intent(android.provider.MediaStore.ACTION_IMAGE_CAPTURE);
                startActivityForResult(Intent.createChooser(pictureIntent, strAvatarPrompt), TAKE_AVATAR_CAMERA_REQUEST);
            }
        });
        
        avatarButton.setOnLongClickListener(new View.OnLongClickListener() {
            
            public boolean onLongClick(View v) {
                String strAvatarPrompt = "アバターに使う画像を選んでください。";
                Intent pickPhoto = new Intent(Intent.ACTION_PICK);
                pickPhoto.setType("image/*");
                startActivityForResult(Intent.createChooser(pickPhoto, strAvatarPrompt), TAKE_AVATAR_GALLERY_REQUEST);
                return true;
            }
        });
    }
    
    /**
     * ニックネーム入力フィールドの初期化
     */
    private void initNicknameEntry() {
        // ニックネームを保存
        final EditText nicknameText = (EditText) findViewById(R.id.EditText_Nickname);
        
        if (mGameSettings.contains(GAME_PREFERENCES_NICKNAME)) {
            nicknameText.setText(mGameSettings.getString(GAME_PREFERENCES_NICKNAME, ""));
        }
        
        nicknameText.setOnKeyListener(new View.OnKeyListener() {
            
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if ((event.getAction() == KeyEvent.ACTION_DOWN) && (keyCode == KeyEvent.KEYCODE_ENTER)) {
                    String strNickname = nicknameText.getText().toString();
                    Editor editor = mGameSettings.edit();
                    editor.putString(GAME_PREFERENCES_NICKNAME, strNickname);
                    editor.commit();
                    return true;
                }
                return false;
            }
        });
    }
    
    /**
     * メールアドレス入力フィールドの初期化
     */
    private void initEmailEntry() {
        // メールアドレスを保存
        final EditText emailText = (EditText) findViewById(R.id.EditText_Email);
        
        if (mGameSettings.contains(GAME_PREFERENCES_EMAIL)) {
            emailText.setText(mGameSettings.getString(GAME_PREFERENCES_EMAIL, ""));
        }
        
        emailText.setOnKeyListener(new View.OnKeyListener() {
            
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if ((event.getAction() == KeyEvent.ACTION_DOWN) && (keyCode == KeyEvent.KEYCODE_ENTER)) {
                    Editor editor = mGameSettings.edit();
                    editor.putString(GAME_PREFERENCES_EMAIL, emailText.getText().toString());
                    editor.commit();
                    return true;
                }
                return false;
            }
        });
        
    }
    
    /**
     * パスワード入力ダイアログの初期化
     */
    private void initPasswordChooser() {
        // パスワード設定状況の表示
        TextView passwordInfo = (TextView) findViewById(R.id.TextView_Password_Info);
        
        if (mGameSettings.contains(GAME_PREFERENCES_PASSWORD)) {
            passwordInfo.setText(R.string.settings_pwd_set);
        } else {
            passwordInfo.setText(R.string.settings_pwd_not_set);
        }
        
        // パスワード入力ダイアログの処理
        Button setPassword = (Button) findViewById(R.id.Button_Password);
        setPassword.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                showDialog(PASSWORD_DIALOG_ID);
            }
        });
    }
    
    /**
     * 生年月日入力用日付ピッカーの初期化
     */
    private void initDatePicker() {
        // 生年月日設定状況の表示
        TextView dobInfo = (TextView) findViewById(R.id.TextView_DOB_Info);
        
        if (mGameSettings.contains(GAME_PREFERENCES_DOB)) {
            dobInfo.setText(DateFormat.format("yyyy年 MMMM dd日", mGameSettings.getLong(GAME_PREFERENCES_DOB, 0)));
        } else {
            dobInfo.setText(R.string.settings_dob_not_set);
        }
        
        // 日付ピッカーダイアログの処理
        Button pickDate = (Button) findViewById(R.id.Button_DOB);
        pickDate.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                showDialog(DATE_DIALOG_ID);
            }
        });
    }
    
    /**
     * 性別を選択するスピナーの初期化
     */
    private void initGenderSpinner() {
        // Spinnerコントロールに性別文字列を設定
        final Spinner spinner = (Spinner) findViewById(R.id.Spinner_Gender);
        ArrayAdapter<?> adapter = ArrayAdapter.createFromResource(this, R.array.genders, android.R.layout.simple_spinner_item);
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        spinner.setAdapter(adapter);
        
        if (mGameSettings.contains(GAME_PREFERENCES_GENDER)) {
            spinner.setSelection(mGameSettings.getInt(GAME_PREFERENCES_GENDER, 0));
        }
        
        // スピナーで選択が行われた際の処理
        spinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            
            public void onItemSelected(AdapterView<?> parent, View itemSelected, int selectedItemPosition, long selectedId) {
                
                Editor editor = mGameSettings.edit();
                editor.putInt(GAME_PREFERENCES_GENDER, selectedItemPosition);
                editor.commit();
            }
            
            public void onNothingSelected(AdapterView<?> parent) {
            }
        });
        
    }
    
    /**
     * 「お気に入りの場所」選択画面の初期化
     */
    private void initFavoritePlacePicker() {
        // 「お気に入りの場所」選択状態の表示
        TextView placeInfo = (TextView) findViewById(R.id.TextView_FavoritePlace_Info);
        
        if (mGameSettings.contains(GAME_PREFERENCES_FAV_PLACE_NAME)) {
            placeInfo.setText(mGameSettings.getString(GAME_PREFERENCES_FAV_PLACE_NAME, ""));
        } else {
            placeInfo.setText(R.string.settings_favoriteplace_not_set);
        }
        
        // 「お気に入りの場所」選択ダイアログの処理
        Button pickPlace = (Button) findViewById(R.id.Button_FavoritePlace);
        pickPlace.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                showDialog(PLACE_DIALOG_ID);
            }
        });
    }
    
    @Override
    protected Dialog onCreateDialog(int id) {
        switch (id) {
            case PLACE_DIALOG_ID:
                
                LayoutInflater layoutInflater = (LayoutInflater) getSystemService(Context.LAYOUT_INFLATER_SERVICE);
                final View dialogLayout = layoutInflater.inflate(R.layout.fav_place_dialog, (ViewGroup) findViewById(R.id.root));
                
                final TextView placeCoordinates = (TextView) dialogLayout.findViewById(R.id.TextView_FavPlaceCoords_Info);
                final EditText placeName = (EditText) dialogLayout.findViewById(R.id.EditText_FavPlaceName);
                placeName.setOnKeyListener(new View.OnKeyListener() {
                    
                    public boolean onKey(View v, int keyCode, KeyEvent event) {
                        if ((event.getAction() == KeyEvent.ACTION_DOWN) && (keyCode == KeyEvent.KEYCODE_ENTER)) {
                            
                            String strPlaceName = placeName.getText().toString();
                            if ((strPlaceName != null) && (strPlaceName.length() > 0)) {
                                // 文字列からGPS座標への変換を試みる
                                resolveLocation(strPlaceName);
                                
                                Editor editor = mGameSettings.edit();
                                editor.putString(GAME_PREFERENCES_FAV_PLACE_NAME, placeName.getText().toString());
                                editor.putFloat(GAME_PREFERENCES_FAV_PLACE_LONG, mFavPlaceCoords.mLon);
                                editor.putFloat(GAME_PREFERENCES_FAV_PLACE_LAT, mFavPlaceCoords.mLat);
                                editor.commit();
                                
                                placeCoordinates.setText(formatCoordinates(mFavPlaceCoords.mLat, mFavPlaceCoords.mLon));
                                return true;
                            }
                        }
                        return false;
                    }
                });
                
                final Button mapButton = (Button) dialogLayout.findViewById(R.id.Button_MapIt);
                mapButton.setOnClickListener(new View.OnClickListener() {
                    public void onClick(View v) {
                        
                        // 文字列からGPS座標への変換を試みる
                        String placeToFind = placeName.getText().toString();
                        resolveLocation(placeToFind);
                        
                        Editor editor = mGameSettings.edit();
                        editor.putString(GAME_PREFERENCES_FAV_PLACE_NAME, placeToFind);
                        editor.putFloat(GAME_PREFERENCES_FAV_PLACE_LONG, mFavPlaceCoords.mLon);
                        editor.putFloat(GAME_PREFERENCES_FAV_PLACE_LAT, mFavPlaceCoords.mLat);
                        editor.commit();
                        
                        placeCoordinates.setText(formatCoordinates(mFavPlaceCoords.mLat, mFavPlaceCoords.mLon));
                        
                        // GPS座標を渡して「マップ」を起動
                        String geoURI = String.format("geo:%f,%f?z=10", mFavPlaceCoords.mLat, mFavPlaceCoords.mLon);
                        Uri geo = Uri.parse(geoURI);
                        Intent geoMap = new Intent(Intent.ACTION_VIEW, geo);
                        startActivity(geoMap);
                    }
                });
                
                AlertDialog.Builder dialogBuilder = new AlertDialog.Builder(this);
                dialogBuilder.setView(dialogLayout);
                
                // AlertDialogの設定
                dialogBuilder.setTitle(R.string.settings_button_favoriteplace);
                
                dialogBuilder.setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int whichButton) {
                        // ダイアログは強制的に終了し、プールから削除するので
                        // 再使用はできない（キャッシュされていない）
                        QuizSettingsActivity.this.removeDialog(PLACE_DIALOG_ID);
                    }
                });
                
                dialogBuilder.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int which) {
                        
                        TextView placeInfo = (TextView) findViewById(R.id.TextView_FavoritePlace_Info);
                        String strPlaceName = placeName.getText().toString();
                        
                        if ((strPlaceName != null) && (strPlaceName.length() > 0)) {
                            Editor editor = mGameSettings.edit();
                            editor.putString(GAME_PREFERENCES_FAV_PLACE_NAME, strPlaceName);
                            editor.putFloat(GAME_PREFERENCES_FAV_PLACE_LONG, mFavPlaceCoords.mLon);
                            editor.putFloat(GAME_PREFERENCES_FAV_PLACE_LAT, mFavPlaceCoords.mLat);
                            editor.commit();
                            
                            placeInfo.setText(strPlaceName);
                        }
                        
                        // ダイアログは強制的に終了し、プールから削除するので
                        // 再使用はできない（キャッシュされていない）
                        QuizSettingsActivity.this.removeDialog(PLACE_DIALOG_ID);
                    }
                });
                
                // AlertDialogを作成し、戻り値として返す。
                AlertDialog placeDialog = dialogBuilder.create();
                return placeDialog;
            case DATE_DIALOG_ID:
                
                final TextView dob = (TextView) findViewById(R.id.TextView_DOB_Info);
                
                DatePickerDialog dateDialog = new DatePickerDialog(this, new DatePickerDialog.OnDateSetListener() {
                    
                    public void onDateSet(DatePicker view, int year, int monthOfYear, int dayOfMonth) {
                        
                        Time dateOfBirth = new Time();
                        dateOfBirth.set(dayOfMonth, monthOfYear, year);
                        long dtDob = dateOfBirth.toMillis(true);
                        dob.setText(DateFormat.format("yyyy年 MMMM dd日", dtDob));
                        
                        Editor editor = mGameSettings.edit();
                        editor.putLong(GAME_PREFERENCES_DOB, dtDob);
                        editor.commit();
                    }
                }, 0, 0, 0);
                return dateDialog;
                
            case PASSWORD_DIALOG_ID:
                
                LayoutInflater inflater = (LayoutInflater) getSystemService(Context.LAYOUT_INFLATER_SERVICE);
                final View layout = inflater.inflate(R.layout.password_dialog, (ViewGroup) findViewById(R.id.root));
                
                final EditText p1 = (EditText) layout.findViewById(R.id.EditText_Pwd1);
                final EditText p2 = (EditText) layout.findViewById(R.id.EditText_Pwd2);
                final TextView error = (TextView) layout.findViewById(R.id.TextView_PwdProblem);
                
                p2.addTextChangedListener(new TextWatcher() {
                    @Override
                    public void afterTextChanged(Editable s) {
                        
                        String strPass1 = p1.getText().toString();
                        String strPass2 = p2.getText().toString();
                        
                        if (strPass1.equals(strPass2)) {
                            error.setText(R.string.settings_pwd_equal);
                        } else {
                            error.setText(R.string.settings_pwd_not_equal);
                        }
                    }
                    
                    // ... 他の必須オーバライドメソッドは何もしない
                    @Override
                    public void beforeTextChanged(CharSequence s, int start, int count, int after) {
                    }
                    
                    @Override
                    public void onTextChanged(CharSequence s, int start, int before, int count) {
                    }
                });
                
                AlertDialog.Builder builder = new AlertDialog.Builder(this);
                builder.setView(layout);
                
                // AlertDialogの設定
                builder.setTitle(R.string.settings_button_pwd);
                
                builder.setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int whichButton) {
                        // ダイアログは強制的に終了し、プールから削除するので
                        // 再使用はできない（キャッシュされていない）
                        QuizSettingsActivity.this.removeDialog(PASSWORD_DIALOG_ID);
                    }
                });
                
                builder.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int which) {
                        
                        TextView passwordInfo = (TextView) findViewById(R.id.TextView_Password_Info);
                        
                        String strPassword1 = p1.getText().toString();
                        String strPassword2 = p2.getText().toString();
                        
                        if (strPassword1.equals(strPassword2)) {
                            Editor editor = mGameSettings.edit();
                            editor.putString(GAME_PREFERENCES_PASSWORD, strPassword1);
                            editor.commit();
                            
                            passwordInfo.setText(R.string.settings_pwd_set);
                        } else {
                            Log.d(DEBUG_TAG, "パスワードが一致していません。保存せず、古いパスワードを維持します（設定されている場合）。");
                        }
                        
                        // ダイアログは強制的に終了し、プールから削除するので
                        // 再使用はできない（キャッシュされていない）
                        QuizSettingsActivity.this.removeDialog(PASSWORD_DIALOG_ID);
                    }
                });
                
                // AlertDialogを作成し、戻り値として返す。
                AlertDialog passwordDialog = builder.create();
                return passwordDialog;
        }
        return null;
    }
    
    @Override
    protected void onPrepareDialog(int id, Dialog dialog) {
        super.onPrepareDialog(id, dialog);
        
        switch (id) {
            case PLACE_DIALOG_ID:
                
                // 「お気に入りの場所」選択ダイアログの初期化処理はすべてここで行う
                AlertDialog placeDialog = (AlertDialog) dialog;
                
                String strFavPlaceName;
                
                // 共有プレファレンスに保存されている「お気に入りの場所」をチェック
                if (mGameSettings.contains(GAME_PREFERENCES_FAV_PLACE_NAME)) {
                    
                    // 共有プレファレンスから「お気に入りの場所」を取得
                    strFavPlaceName = mGameSettings.getString(GAME_PREFERENCES_FAV_PLACE_NAME, "");
                    mFavPlaceCoords = new GPSCoords(mGameSettings.getFloat(GAME_PREFERENCES_FAV_PLACE_LAT, 0), mGameSettings.getFloat(GAME_PREFERENCES_FAV_PLACE_LONG, 0));
                    
                } else {
                    
                    // 「お気に入りの場所」が設定されていないので、現在地の座標を設定
                    strFavPlaceName = getResources().getString(R.string.settings_favplace_currentlocation);
                    // 現在地には名前（「ここ」）を付けずに地図上の一点として扱う。
                    // ユーザーが名前を付けたければ、付けることができる。
                    calculateCurrentCoordinates();
                    
                }
                
                // 場所の名前と座標を設定する。
                // 保存してある値があればその値に設定し、保存してある値がなければ、現在地のGPS座標に設定する。
                final EditText placeName = (EditText) placeDialog.findViewById(R.id.EditText_FavPlaceName);
                placeName.setText(strFavPlaceName);
                
                final TextView placeCoordinates = (TextView) placeDialog.findViewById(R.id.TextView_FavPlaceCoords_Info);
                placeCoordinates.setText(formatCoordinates(mFavPlaceCoords.mLat, mFavPlaceCoords.mLon));
                
                return;
            case DATE_DIALOG_ID:
                
                // DatePickerDialogの初期化処理はすべてここで行う
                DatePickerDialog dateDialog = (DatePickerDialog) dialog;
                int iDay,
                iMonth,
                iYear;
                
                // プレファレンス情報に生年月日データがあるか調べる
                if (mGameSettings.contains(GAME_PREFERENCES_DOB)) {
                    // プレファレンス情報から生年月日を取り出して設定
                    long msBirthDate = mGameSettings.getLong(GAME_PREFERENCES_DOB, 0);
                    Time dateOfBirth = new Time();
                    dateOfBirth.set(msBirthDate);
                    
                    iDay = dateOfBirth.monthDay;
                    iMonth = dateOfBirth.month;
                    iYear = dateOfBirth.year;
                } else {
                    Calendar cal = Calendar.getInstance();
                    
                    // 今日の日付を取り出して設定
                    iDay = cal.get(Calendar.DAY_OF_MONTH);
                    iMonth = cal.get(Calendar.MONTH);
                    iYear = cal.get(Calendar.YEAR);
                }
                
                // DatePickerの日付を生年月日、または
                // 現在日付に設定する
                dateDialog.updateDate(iYear, iMonth, iDay);
                return;
                
            case PASSWORD_DIALOG_ID:
                // パスワードダイアログの初期化があればすべてここで処理する。
                // このサンプルプログラムでは、一度表示したパスワードダイアログは再表示せず、
                // 新しく設定するので、ここでする作業はない。
                // 使い終わったパスワードダイアログは、removeDialog()を呼び出して
                // アクティビティのダイアログプールから削除している。
                // パスワードダイアログが必要になった場合は、生成し直す。
                return;
        }
    }
    
    /**
     * GPS座標値を画面に表示する書式にフォーマットするヘルパーメソッド
     * 
     * @param lat
     * @param lon
     * @return フォーマット済みの文字列
     */
    private String formatCoordinates(float lat, float lon) {
        StringBuilder strCoords = new StringBuilder();
        strCoords.append(lat).append(",").append(lon);
        return strCoords.toString();
    }
    
    /**
     * 
     * 場所名から場所が決定できなかった場合にGPS座標から場所を決定する
     * 
     * @param strLocation
     *            場所を決定したい場所名や地名
     */
    private void resolveLocation(String strLocation) {
        boolean bResolvedAddress = false;
        
        if (strLocation.equalsIgnoreCase(getResources().getString(R.string.settings_favplace_currentlocation)) == false) {
            bResolvedAddress = lookupLocationByName(strLocation);
        }
        
        if (bResolvedAddress == false) {
            // Stringオブジェクトで与えられた場所名から場所が決定できない場合（あるいは場所名が「(現在位置)」の場合、この場所名はユーザーが独自につけたものと見なす。
            calculateCurrentCoordinates();
        }
    }
    
    /**
     * 機器の最終決定位置（last known location）の取得を試みる。
     * この値は位置情報プロバイダが最後に設定したものであるのが普通。
     */
    private void calculateCurrentCoordinates() {
        float lat = 0, lon = 0;
        
        try {
            LocationManager locMgr = (LocationManager) getSystemService(LOCATION_SERVICE);
            Location recentLoc = locMgr.getLastKnownLocation(LocationManager.GPS_PROVIDER);
            lat = (float) recentLoc.getLatitude();
            lon = (float) recentLoc.getLongitude();
        } catch (Exception e) {
            Log.e(DEBUG_TAG, "位置情報の処理に失敗しました", e);
        }
        
        mFavPlaceCoords = new GPSCoords(lat, lon);
    }
    
    /**
     * 
     * 場所の名称や住所などを文字列で受け取り、得られた座標を{@code mFavPlaceCoords}に保存
     * 
     * @param strLocation
     *            検索する場所名や地名
     * @return 場所名や住所が認識できれば{@code true}、できなければ{@code false}
     */
    private boolean lookupLocationByName(String strLocation) {
        final Geocoder coder = new Geocoder(getApplicationContext());
        boolean bResolvedAddress = false;
        
        try {
            
            List<Address> geocodeResults = coder.getFromLocationName(strLocation, 1);
            Iterator<Address> locations = geocodeResults.iterator();
            
            while (locations.hasNext()) {
                Address loc = locations.next();
                mFavPlaceCoords = new GPSCoords((float) loc.getLatitude(), (float) loc.getLongitude());
                bResolvedAddress = true;
            }
        } catch (Exception e) {
            Log.e(DEBUG_TAG, "場所名のジオコーディングに失敗しました", e);
        }
        return bResolvedAddress;
    }
    
    private class GPSCoords {
        float mLat, mLon;
        
        GPSCoords(float lat, float lon) {
            mLat = lat;
            mLon = lon;
            
        }
    }
}
