package com.androidbook.triviaquiz19;

import java.io.IOException;
import java.net.URL;
import java.util.Hashtable;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.DialogInterface.OnCancelListener;
import android.content.SharedPreferences.Editor;
import android.content.res.Resources;
import android.content.res.XmlResourceParser;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.os.AsyncTask;
import android.os.Bundle;
import android.util.Log;
import android.view.Gravity;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup.LayoutParams;
import android.view.animation.Animation;
import android.view.animation.AnimationUtils;
import android.widget.Button;
import android.widget.ImageSwitcher;
import android.widget.ImageView;
import android.widget.TextSwitcher;
import android.widget.TextView;
import android.widget.ViewSwitcher;

public class QuizGameActivity extends QuizActivity {
    
    /** アクティビティが最初に生成されるときに呼び出される */
    
    SharedPreferences mGameSettings;
    Hashtable<Integer, Question> mQuestions;
    private TextSwitcher mQuestionText;
    private ImageSwitcher mQuestionImage;
    QuizTask downloader;
    
    @Override
    protected void onPause() {
        if (downloader != null && downloader.getStatus() != AsyncTask.Status.FINISHED) {
            downloader.cancel(true);
        }
        super.onPause();
    }
    
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.game);
        
        // 共有プレファレンスの取得
        mGameSettings = getSharedPreferences(GAME_PREFERENCES, Context.MODE_PRIVATE);
        
        // 問題集の初期化
        mQuestions = new Hashtable<Integer, Question>(QUESTION_BATCH_SIZE);
        
        // 次に回答する問題番号の読み込み
        int startingQuestionNumber = mGameSettings.getInt(GAME_PREFERENCES_CURRENT_QUESTION, 0);
        
        // ゲームを開始したところであれば、共有プレファレンスを初期化する
        if (startingQuestionNumber == 0) {
            startingQuestionNumber = 1;
            Editor editor = mGameSettings.edit();
            editor.putInt(GAME_PREFERENCES_CURRENT_QUESTION, startingQuestionNumber);
            editor.commit();
        }
        
        // 問題集のダウンロードをバックグラウンドで開始
        downloader = new QuizTask();
        downloader.execute(TRIVIA_SERVER_QUESTIONS, startingQuestionNumber);
        
        // 問題集のダウンロードと平行してUIの設定
        // ［はい］ボタンの処理
        Button yesButton = (Button) findViewById(R.id.Button_Yes);
        yesButton.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                handleAnswerAndShowNextQuestion(true);
            }
        });
        
        // ［いいえ］ボタンの処理
        Button noButton = (Button) findViewById(R.id.Button_No);
        noButton.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                handleAnswerAndShowNextQuestion(false);
            }
        });
        
        // TextSwitcherコントロールとImageSwitcherコントロールの初期設定
        Animation in = AnimationUtils.loadAnimation(this, android.R.anim.fade_in);
        Animation out = AnimationUtils.loadAnimation(this, android.R.anim.fade_out);
        
        mQuestionText = (TextSwitcher) findViewById(R.id.TextSwitcher_QuestionText);
        mQuestionText.setInAnimation(in);
        mQuestionText.setOutAnimation(out);
        mQuestionText.setFactory(new MyTextSwitcherFactory());
        
        mQuestionImage = (ImageSwitcher) findViewById(R.id.ImageSwitcher_QuestionImage);
        mQuestionImage.setInAnimation(in);
        mQuestionImage.setOutAnimation(out);
        mQuestionImage.setFactory(new MyImageSwitcherFactory());
    }
    
    /**
     * 問題集のダウンロードが完了したら呼び出される
     * 
     * @param startingQuestionNumber
     *            問題集の最初の問題番号
     */
    private void displayCurrentQuestion(int startingQuestionNumber) {
        // 問題が正しく読み込めたら、問題を表示
        if (mQuestions.containsKey(startingQuestionNumber) == true) {
            // TextSwitcherに問題文を設定
            mQuestionText.setCurrentText(getQuestionText(startingQuestionNumber));
            
            // ImageSwitcherに問題画像を設定
            Drawable image = getQuestionImageDrawable(startingQuestionNumber);
            mQuestionImage.setImageDrawable(image);
        } else {
            // この時点でもう表示できる問題データがないことをユーザーに伝達
            handleNoQuestions();
        }
    }
    
    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        super.onCreateOptionsMenu(menu);
        
        getMenuInflater().inflate(R.menu.gameoptions, menu);
        menu.findItem(R.id.help_menu_item).setIntent(new Intent(this, QuizHelpActivity.class));
        menu.findItem(R.id.settings_menu_item).setIntent(new Intent(this, QuizSettingsActivity.class));
        return true;
    }
    
    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        super.onOptionsItemSelected(item);
        startActivity(item.getIntent());
        return true;
    }
    
    /**
     * 問題画像に使う{@code SwitcherFactory}コントロール。
     * アニメーションで遷移する先の{@code ImageView}オブジェクトを生成する。
     * 
     */
    private class MyImageSwitcherFactory implements ViewSwitcher.ViewFactory {
        public View makeView() {
            ImageView imageView = new ImageView(QuizGameActivity.this);
            imageView.setScaleType(ImageView.ScaleType.FIT_CENTER);
            imageView.setLayoutParams(new ImageSwitcher.LayoutParams(LayoutParams.FILL_PARENT, LayoutParams.FILL_PARENT));
            return imageView;
        }
    }
    
    /**
     * 問題文に使う{@code SwitcherFactory}コントロール。
     * アニメーションで遷移する先の{@code TextView}オブジェクトを生成する。
     * 
     */
    private class MyTextSwitcherFactory implements ViewSwitcher.ViewFactory {
        public View makeView() {
            TextView textView = new TextView(QuizGameActivity.this);
            textView.setGravity(Gravity.CENTER);
            Resources res = getResources();
            float dimension = res.getDimension(R.dimen.game_question_size);
            int titleColor = res.getColor(R.color.title_color);
            int shadowColor = res.getColor(R.color.title_glow);
            textView.setTextSize(dimension);
            textView.setTextColor(titleColor);
            textView.setShadowLayer(10, 5, 5, shadowColor);
            return textView;
        }
    }
    
    /**
     * 
     * ユーザーの回答を記録し、次の問題を読み込むヘルパーメソッド
     * 
     * @param bAnswer
     *            ユーザーの回答
     */
    private void handleAnswerAndShowNextQuestion(boolean bAnswer) {
        int curScore = mGameSettings.getInt(GAME_PREFERENCES_SCORE, 0);
        int nextQuestionNumber = mGameSettings.getInt(GAME_PREFERENCES_CURRENT_QUESTION, 1) + 1;
        
        Editor editor = mGameSettings.edit();
        editor.putInt(GAME_PREFERENCES_CURRENT_QUESTION, nextQuestionNumber);
        
        // 「はい」と回答した数だけを記録
        if (bAnswer == true) {
            editor.putInt(GAME_PREFERENCES_SCORE, curScore + 1);
        }
        editor.commit();
        
        if (mQuestions.containsKey(nextQuestionNumber) == false) {
            
            downloader = new QuizTask();
            downloader.execute(TRIVIA_SERVER_QUESTIONS, nextQuestionNumber);
            
            // ダウンロードが完了するまで問題の表示を延期
        } else {
            
            displayCurrentQuestion(nextQuestionNumber);
        }
    }
    
    /**
     * 問題がもう残っていない場合にゲーム画面の表示を変更するヘルパーメソッド
     * さまざまなエラーが発生した場合に使えるので、
     * 問題のデータがない場合、IOエラー、パーサエラーなどで呼び出してもよい。
     */
    private void handleNoQuestions() {
        TextSwitcher questionTextSwitcher = (TextSwitcher) findViewById(R.id.TextSwitcher_QuestionText);
        questionTextSwitcher.setText(getResources().getText(R.string.no_questions));
        ImageSwitcher questionImageSwitcher = (ImageSwitcher) findViewById(R.id.ImageSwitcher_QuestionImage);
        questionImageSwitcher.setImageResource(R.drawable.noquestion);
        
        // ［はい］ボタンを無効化
        Button yesButton = (Button) findViewById(R.id.Button_Yes);
        yesButton.setEnabled(false);
        
        // ［いいえ］ボタンを無効化
        Button noButton = (Button) findViewById(R.id.Button_No);
        noButton.setEnabled(false);
    }
    
    /**
     * 指定された問題番号に対応する問題文を入れた{@code String}を返す
     * 
     * @param questionNumber
     *            問題文を取得したい問題の番号
     * @return 問題文。問題番号{@code questionNumber}の問題が見つからなければnull。
     */
    private String getQuestionText(Integer questionNumber) {
        String text = null;
        Question curQuestion = (Question) mQuestions.get(questionNumber);
        if (curQuestion != null) {
            text = curQuestion.mText;
        }
        return text;
    }
    
    /**
     * 指定された問題番号に対応する問題画像のURLを入れた{@code String}を返す
     * 
     * @param questionNumber
     *            問題画像のURLを取得したい問題の番号
     * @return URLを入れた{@code String}。問題が見つからなければnull。
     */
    private String getQuestionImageUrl(Integer questionNumber) {
        String url = null;
        Question curQuestion = (Question) mQuestions.get(questionNumber);
        if (curQuestion != null) {
            url = curQuestion.mImageUrl;
        }
        return url;
    }
    
    /**
     * 指定された問題の{@code Drawable}オブジェクトを取得する
     * 
     * @param questionNumber
     *            {@code Drawable}オブジェクトを取得したい問題の番号
     * @return 指定された問題の{@code Drawable}オブジェクト。読み込みに失敗した場合や問題番号に相当する問題がない場合はダミーの画像
     */
    private Drawable getQuestionImageDrawable(int questionNumber) {
        Drawable image;
        URL imageUrl;
        
        try {
            // リモートURLからのストリームをデコードしてDrawableを生成する
            imageUrl = new URL(getQuestionImageUrl(questionNumber));
            Bitmap bitmap = BitmapFactory.decodeStream(imageUrl.openStream());
            image = new BitmapDrawable(bitmap);
        } catch (Exception e) {
            Log.e(DEBUG_TAG, "ビットマップストリームのデコードに失敗しました。");
            image = getResources().getDrawable(R.drawable.noquestion);
        }
        return image;
    }
    
    /**
     * 個々の問題データを管理するためのオブジェクト
     * 
     */
    private class Question {
        @SuppressWarnings("unused")
        int mNumber;
        String mText;
        String mImageUrl;
        
        /**
         * 
         * 新しい{@code Question}オブジェクトを生成するコンストラクタ
         * 
         * @param questionNum
         *            この問題の問題番号
         * @param questionText
         *            この問題の問題文
         * @param questionImageUrl
         *            この問題に表示する画像の有効なURL
         */
        public Question(int questionNum, String questionText, String questionImageUrl) {
            mNumber = questionNum;
            mText = questionText;
            mImageUrl = questionImageUrl;
        }
    }
    
    private class QuizTask extends AsyncTask<Object, String, Boolean> {
        private static final String DEBUG_TAG = "QuizGameActivity$QuizTask";
        
        int startingNumber;
        ProgressDialog pleaseWaitDialog;
        
        @Override
        protected void onCancelled() {
            Log.i(DEBUG_TAG, "onCancelled");
            handleNoQuestions();
            pleaseWaitDialog.dismiss();
        }
        
        @Override
        protected void onPostExecute(Boolean result) {
            Log.d(DEBUG_TAG, "ダウンロード完了。");
            if (result) {
                displayCurrentQuestion(startingNumber);
            } else {
                handleNoQuestions();
            }
            
            pleaseWaitDialog.dismiss();
        }
        
        @Override
        protected void onPreExecute() {
            pleaseWaitDialog = ProgressDialog.show(QuizGameActivity.this, "旅の達人", "問題集のダウンロード", true, true);
            pleaseWaitDialog.setOnCancelListener(new OnCancelListener() {
                public void onCancel(DialogInterface dialog) {
                    QuizTask.this.cancel(true);
                }
            });
        }
        
        @Override
        protected void onProgressUpdate(String... values) {
            super.onProgressUpdate(values);
        }
        
        @Override
        protected Boolean doInBackground(Object... params) {
            boolean result = false;
            try {
                // パラメータは正しい順序で、正しい型のものを付加する。誤ったものを付加すると例外ClassCastExceptionが発行される
                startingNumber = (Integer) params[1];
                String pathToQuestions = params[0] + "?max=" + QUESTION_BATCH_SIZE + "&start=" + startingNumber;
                
                // アカウントが登録されていれば得点情報を更新。
                // 反応の遅れを少なくし、ネットワーク効率を向上させるため、同じリクエストの中で行う。
                SharedPreferences settings = getSharedPreferences(GAME_PREFERENCES, Context.MODE_PRIVATE);
                
                Integer playerId = settings.getInt(GAME_PREFERENCES_PLAYER_ID, -1);
                if (playerId != -1) {
                    Log.d(DEBUG_TAG, "得点情報の更新");
                    Integer score = settings.getInt(GAME_PREFERENCES_SCORE, -1);
                    if (score != -1) {
                        pathToQuestions += "&updateScore=yes&updateId=" + playerId + "&score=" + score;
                    }
                }
                
                Log.d(DEBUG_TAG, "path: " + pathToQuestions + " -- Num: " + startingNumber);
                
                result = loadQuestionBatch(startingNumber, pathToQuestions);
                
            } catch (Exception e) {
                Log.e(DEBUG_TAG, "XMLをダウンロードしパースしているときに予期せず失敗しました", e);
            }
            
            return result;
        }
        
        /**
         * XML形式の問題集データをパースし{@see mQuestions}に保存する。問題集はXmlPullParser (questionBatch)に読み込まれている。
         * 
         * @param questionBatch
         *            データ読み出し用のXmlPullParser
         * @throws XmlPullParserException
         *             XMLパースエラーで発行される
         * @throws IOException
         *             XML読み込みのエラーで発行される
         */
        private void parseXMLQuestionBatch(XmlPullParser questionBatch) throws XmlPullParserException, IOException {
            int eventType = -1;
            
            // XMLデータ中の問題のレコードを見つける
            while (eventType != XmlResourceParser.END_DOCUMENT) {
                if (eventType == XmlResourceParser.START_TAG) {
                    
                    // タグ名を取得する（例　questionsまたはquestion）
                    String strName = questionBatch.getName();
                    
                    if (strName.equals(XML_TAG_QUESTION)) {
                        
                        String questionNumber = questionBatch.getAttributeValue(null, XML_TAG_QUESTION_ATTRIBUTE_NUMBER);
                        Integer questionNum = new Integer(questionNumber);
                        String questionText = questionBatch.getAttributeValue(null, XML_TAG_QUESTION_ATTRIBUTE_TEXT);
                        String questionImageUrl = questionBatch.getAttributeValue(null, XML_TAG_QUESTION_ATTRIBUTE_IMAGEURL);
                        
                        // Hashtable型メンバ変数にデータを格納する
                        mQuestions.put(questionNum, new Question(questionNum, questionText, questionImageUrl));
                    }
                }
                eventType = questionBatch.next();
            }
        }
        
        /**
         * メンバ変数{@see mQuestions}にXMLを読み込む
         * 
         * @param startQuestionNumber
         *            読み込みを開始する問題の番号
         */
        private boolean loadQuestionBatch(int startQuestionNumber, String xmlSource) {
            boolean result = false;
            // 現在の（古い）問題集を破棄
            mQuestions.clear();
            
            // サーバーに接続し、startQuestionNumberで始まる問題集データを取得
            XmlPullParser questionBatch;
            try {
                URL xmlUrl = new URL(xmlSource);
                questionBatch = XmlPullParserFactory.newInstance().newPullParser();
                questionBatch.setInput(xmlUrl.openStream(), null);
            } catch (XmlPullParserException e1) {
                questionBatch = null;
                Log.e(DEBUG_TAG, "XmlPullParserの初期化に失敗", e1);
            } catch (IOException e) {
                questionBatch = null;
                Log.e(DEBUG_TAG, "XmlPullParser初期化中のIOエラー", e);
            }
            
            // Parse the XML
            if (questionBatch != null) {
                try {
                    parseXMLQuestionBatch(questionBatch);
                    result = true;
                } catch (XmlPullParserException e) {
                    Log.e(DEBUG_TAG, "XmlPullParserのエラー", e);
                } catch (IOException e) {
                    Log.e(DEBUG_TAG, "XMLパース中のIOエラー", e);
                }
            }
            
            return result;
            
        }
        
    }
}
