package com.androidbook.triviaquiz19;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import org.xmlpull.v1.XmlPullParser;
import org.xmlpull.v1.XmlPullParserException;
import org.xmlpull.v1.XmlPullParserFactory;

import android.app.PendingIntent;
import android.app.Service;
import android.appwidget.AppWidgetManager;
import android.appwidget.AppWidgetProvider;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.IBinder;
import android.util.Log;
import android.widget.RemoteViews;

public class QuizWidgetProvider extends AppWidgetProvider {

    // private static final int IO_BUFFER_SIZE = 4 * 1024;

    @Override
    public void onUpdate(Context context, AppWidgetManager appWidgetManager, int[] appWidgetIds) {
        // バックグラウンドで実行されるようにサービスに入れ込む。
        // WidgetProviderが廃棄される可能性があるので、スレッドを使うことはできない。
        // （AndroidManifestにサービスに関する項目を追加するのを忘れないこと）

        Intent serviceIntent = new Intent(context, WidgetUpdateService.class);
        context.startService(serviceIntent);

    }

    @Override
    public void onDeleted(Context context, int[] appWidgetIds) {
        // 原注: appWidgetIdsを無視してもこのサンプルプログラムの場合は問題ない。
        // しかし、このウィジットのインスタンスがひとつ以上実行されている場合は更新が行われない可能性がある。
        // このウィジットは複数のインスタンスが実行される仕様になっていない。
        Intent serviceIntent = new Intent(context, WidgetUpdateService.class);
        context.stopService(serviceIntent);

        super.onDeleted(context, appWidgetIds);

    }

    public static class WidgetUpdateService extends Service {
        Thread widgetUpdateThread = null;

        private static final String DEBUG_TAG = "WidgetUpdateService";

        @Override
        public int onStartCommand(Intent intent, int flags, final int startId) {
            widgetUpdateThread = new Thread() {
                public void run() {
                    Context context = WidgetUpdateService.this;
                    WidgetData widgetData = new WidgetData("不明", "該当せず", "");

                    getWidgetData(widgetData);

                    // RemoteViewの準備
                    String packageName = context.getPackageName();
                    Log.d(DEBUG_TAG, "packageName: " + packageName);
                    RemoteViews remoteView = new RemoteViews(context.getPackageName(), R.layout.widget);
                    remoteView.setTextViewText(R.id.widget_nickname, widgetData.nickname);
                    remoteView.setTextViewText(R.id.widget_score, "得点: " + widgetData.score);
                    if (widgetData.avatarUrl.length() > 0) {
                        // remoteView.setImageViewUri(R.id.widget_image, Uri.parse(avatarUrl));
                        URL image;
                        try {
                            image = new URL(widgetData.avatarUrl);
                            Log.d(DEBUG_TAG, "avatarUrl: " + widgetData.avatarUrl);

                            // decodeStreamを直接使わない理由については http://bit.ly/bAtW6W と http://bit.ly/a3Qkw4 を参照。
                            // （簡単に言えば、動くことは動くが、特定の状況では動かない）
                            // 以下に示した回避策は『Android Wireless Application Development』でも採用されている。

                            Bitmap bitmap = BitmapFactory.decodeStream(image.openStream());
                            /*
                             * BufferedInputStream in;
                             * BufferedOutputStream out;
                             * 
                             * in = new BufferedInputStream(image.openStream(), IO_BUFFER_SIZE);
                             * final ByteArrayOutputStream dataStream = new ByteArrayOutputStream();
                             * out = new BufferedOutputStream(dataStream, IO_BUFFER_SIZE);
                             * copy(in, out);
                             * // 実装はこのファイルの最後。使用する場合はコメントを解除すること。
                             * out.flush();
                             * 
                             * final byte[] data = dataStream.toByteArray();
                             * Log.d(DEBUG_TAG, "Length: "+ data.length);
                             * Bitmap bitmap = BitmapFactory.decodeByteArray(data, 0, data.length);
                             */
                            if (bitmap == null) {
                                Log.w(DEBUG_TAG, "画像のデコードに失敗しました");

                                remoteView.setImageViewResource(R.id.widget_image, R.drawable.avatar);
                            } else {
                                remoteView.setImageViewBitmap(R.id.widget_image, bitmap);
                            }
                        } catch (MalformedURLException e) {
                            Log.e(DEBUG_TAG, "画像のURLエラー", e);
                        } catch (IOException e) {
                            Log.e(DEBUG_TAG, "画像のIOエラー", e);
                        }

                    } else {
                        remoteView.setImageViewResource(R.id.widget_image, R.drawable.avatar);
                    }

                    try {

                        // クリックの処理を追加
                        Intent launchAppIntent = new Intent(context, QuizMenuActivity.class);
                        PendingIntent launchAppPendingIntent = PendingIntent.getActivity(context, 0, launchAppIntent, PendingIntent.FLAG_UPDATE_CURRENT);
                        remoteView.setOnClickPendingIntent(R.id.widget_view, launchAppPendingIntent);

                        // QuizWidgetProviderに対するAndroidコンポーネント名を取得
                        ComponentName quizWidget = new ComponentName(context, QuizWidgetProvider.class);

                        // AppWidgetManagerのインスタンスを取得
                        AppWidgetManager appWidgetManager = AppWidgetManager.getInstance(context);

                        // ウィジットの更新
                        appWidgetManager.updateAppWidget(quizWidget, remoteView);

                    } catch (Exception e) {
                        Log.e(DEBUG_TAG, "ウィジットの更新に失敗しました", e);
                    }

                    if (!WidgetUpdateService.this.stopSelfResult(startId)) {
                        Log.e(DEBUG_TAG, "サービスを停止できませんでした");
                    }
                }

                /**
                 * ウィジットに表示するデータのダウンロード
                 * 
                 * @param widgetData
                 */
                private void getWidgetData(WidgetData widgetData) {
                    SharedPreferences prefs = getSharedPreferences(QuizActivity.GAME_PREFERENCES, Context.MODE_PRIVATE);
                    Integer playerId = prefs.getInt(QuizActivity.GAME_PREFERENCES_PLAYER_ID, -1);

                    try {
                        URL userInfo = new URL(QuizActivity.TRIVIA_SERVER_BASE + "getplayer?playerId=" + playerId);
                        XmlPullParser parser = XmlPullParserFactory.newInstance().newPullParser();
                        parser.setInput(userInfo.openStream(), null);

                        int eventType = -1;
                        while (eventType != XmlPullParser.END_DOCUMENT) {
                            if (eventType == XmlPullParser.START_TAG) {
                                String strName = parser.getName();

                                if (strName.equals("nickname")) {
                                    widgetData.nickname = parser.nextText();
                                } else if (strName.equals("score")) {
                                    widgetData.score = parser.nextText();
                                } else if (strName.equals("avatarUrl")) {
                                    widgetData.avatarUrl = parser.nextText();
                                }
                            }
                            eventType = parser.next();
                        }
                    } catch (MalformedURLException e) {
                        Log.e(DEBUG_TAG, "URLのエラー", e);
                    } catch (XmlPullParserException e) {
                        Log.e(DEBUG_TAG, "パーサのエラー", e);
                    } catch (IOException e) {
                        Log.e(DEBUG_TAG, "IOエラー", e);
                    }
                }
            };

            // バックグラウンドのスレッドを起動
            widgetUpdateThread.start();

            // ウィジットが消去された場合、元のインテントを使って再起動する。
            // そうすれば追加データが再度送られるので、あとで使うこともできる
            return START_REDELIVER_INTENT;
        }

        @Override
        public void onDestroy() {
            widgetUpdateThread.interrupt();
            super.onDestroy();
        }

        @Override
        public IBinder onBind(Intent intent) {
            // バインドはしない。App Widgetからはバインドできない
            return null;
        }

        /**
         * 入力ストリームの内容を出力ストリームにコピー
         * バイト配列を一時的に割り当てて作業用バッファーとして使う。バッファーの大きさは{@link #IO_BUFFER_SIZE}により定義される。
         * 
         * @param in
         *            コピー元の入力ストリーム
         * @param out
         *            コピー先の出力ストリーム
         * @throws IOException
         *             コピー中に何らかのエラーが起こると発行される
         */
        /*
         * private static void copy(InputStream in, OutputStream out) throws IOException { byte[] b = new byte[IO_BUFFER_SIZE]; int read; while ((read = in.read(b)) != -1) { out.write(b, 0, read); } }
         */

        public static class WidgetData {
            String nickname;
            String score;
            String avatarUrl;

            public WidgetData(String nickname, String score, String avatarUrl) {
                super();
                this.nickname = nickname;
                this.score = score;
                this.avatarUrl = avatarUrl;
            }

            public String getNickname() {
                return nickname;
            }

            public void setNickname(String nickname) {
                this.nickname = nickname;
            }

            public String getScore() {
                return score;
            }

            public void setScore(String score) {
                this.score = score;
            }

            public String getAvatarUrl() {
                return avatarUrl;
            }

            public void setAvatarUrl(String avatarUrl) {
                this.avatarUrl = avatarUrl;
            }
        }
    }

}
