package com.androidbook.triviaquiz10;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.os.Bundle;
import android.text.format.DateFormat;
import android.util.Log;
import android.view.KeyEvent;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.Toast;

public class QuizSettingsActivity extends QuizActivity {
    SharedPreferences mGameSettings;

    /** アクティビティが最初に生成されるときに呼び出される */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.settings);
        // 共有プレファレンスの読み込み
        mGameSettings = getSharedPreferences(GAME_PREFERENCES, Context.MODE_PRIVATE);
        // ニックネーム入力フィールドの初期化
        initNicknameEntry();
        // メールアドレス入力フィールドの初期化
        initEmailEntry();
        // パスワード入力ダイアログの初期化
        initPasswordChooser();
        // 生年月日入力用日付ピッカーの初期化
        initDatePicker();
        // 性別を選択するスピナーの初期化
        initGenderSpinner();
    }

    @Override
    protected void onDestroy() {
        Log.d(DEBUG_TAG, "SHARED PREFERENCES");
        Log.d(DEBUG_TAG, "ニックネーム: " + mGameSettings.getString(GAME_PREFERENCES_NICKNAME, "未設定"));
        Log.d(DEBUG_TAG, "メールアドレス: " + mGameSettings.getString(GAME_PREFERENCES_EMAIL, "未設定"));
        Log.d(DEBUG_TAG, "性別（男=1, 女=2, 秘密=0）: " + mGameSettings.getInt(GAME_PREFERENCES_GENDER, 0));
        // 現時点のプログラムでは、パスワードはまだ保存していない
        Log.d(DEBUG_TAG, "パスワード: " + mGameSettings.getString(GAME_PREFERENCES_PASSWORD, "未設定"));
        // 現時点のプログラムでは、生年月日はまだ保存していない
        Log.d(DEBUG_TAG, "生年月日: "
                + DateFormat.format("yyyy年 MMMM dd日", mGameSettings.getLong(GAME_PREFERENCES_DOB, 0)));
        super.onDestroy();
    }

    /**
     * ニックネーム入力フィールドの初期化
     */
    private void initNicknameEntry() {
        // ニックネームを保存
        final EditText nicknameText = (EditText) findViewById(R.id.EditText_Nickname);
        if (mGameSettings.contains(GAME_PREFERENCES_NICKNAME)) {
            nicknameText.setText(mGameSettings.getString(GAME_PREFERENCES_NICKNAME, ""));
        }
        nicknameText.setOnKeyListener(new View.OnKeyListener() {
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if ((event.getAction() == KeyEvent.ACTION_DOWN) && (keyCode == KeyEvent.KEYCODE_ENTER)) {
                    String strNickname = nicknameText.getText().toString();
                    Editor editor = mGameSettings.edit();
                    editor.putString(GAME_PREFERENCES_NICKNAME, strNickname);
                    editor.commit();
                    return true;
                }
                return false;
            }
        });
    }

    /**
     * メールアドレス入力フィールドの初期化
     */
    private void initEmailEntry() {
        // メールアドレスを保存
        final EditText emailText = (EditText) findViewById(R.id.EditText_Email);
        if (mGameSettings.contains(GAME_PREFERENCES_EMAIL)) {
            emailText.setText(mGameSettings.getString(GAME_PREFERENCES_EMAIL, ""));
        }
        emailText.setOnKeyListener(new View.OnKeyListener() {
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if ((event.getAction() == KeyEvent.ACTION_DOWN) && (keyCode == KeyEvent.KEYCODE_ENTER)) {
                    Editor editor = mGameSettings.edit();
                    editor.putString(GAME_PREFERENCES_EMAIL, emailText.getText().toString());
                    editor.commit();
                    return true;
                }
                return false;
            }
        });
    }

    /**
     * パスワード入力ダイアログの初期化
     */
    private void initPasswordChooser() {
        // パスワード設定状況の表示
        TextView passwordInfo = (TextView) findViewById(R.id.TextView_Password_Info);
        if (mGameSettings.contains(GAME_PREFERENCES_PASSWORD)) {
            passwordInfo.setText(R.string.settings_pwd_set);
        } else {
            passwordInfo.setText(R.string.settings_pwd_not_set);
        }
        // パスワード入力ダイアログの処理
        Button setPassword = (Button) findViewById(R.id.Button_Password);
        setPassword.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                Toast.makeText(QuizSettingsActivity.this, "TODO: パスワード入力ダイアログを起動", Toast.LENGTH_LONG).show();
            }
        });
    }

    /**
     * 生年月日入力用日付ピッカーの初期化
     */
    private void initDatePicker() {
        // 生年月日設定状況の表示
        TextView dobInfo = (TextView) findViewById(R.id.TextView_DOB_Info);
        if (mGameSettings.contains(GAME_PREFERENCES_DOB)) {
            dobInfo.setText(DateFormat.format("yyyy年 MMMM dd日", mGameSettings.getLong(GAME_PREFERENCES_DOB, 0)));
        } else {
            dobInfo.setText(R.string.settings_dob_not_set);
        }
        // 日付ピッカーダイアログの処理
        Button pickDate = (Button) findViewById(R.id.Button_DOB);
        pickDate.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                Toast.makeText(QuizSettingsActivity.this, "TODO: 日付ピッカー（DatePickerDialog）を起動", Toast.LENGTH_LONG).show();
            }
        });
    }

    /**
     * 性別を選択するスピナーの初期化
     */
    private void initGenderSpinner() {
        // Spinnerコントロールに性別文字列を設定
        final Spinner spinner = (Spinner) findViewById(R.id.Spinner_Gender);
        ArrayAdapter<?> adapter = ArrayAdapter.createFromResource(this, R.array.genders,
                android.R.layout.simple_spinner_item);
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        spinner.setAdapter(adapter);
        if (mGameSettings.contains(GAME_PREFERENCES_GENDER)) {
            spinner.setSelection(mGameSettings.getInt(GAME_PREFERENCES_GENDER, 0));
        }
        // スピナーによる選択の処理
        spinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            public void onItemSelected(AdapterView<?> parent, View itemSelected, int selectedItemPosition,
                    long selectedId) {
                Editor editor = mGameSettings.edit();
                editor.putInt(GAME_PREFERENCES_GENDER, selectedItemPosition);
                editor.commit();
            }

            public void onNothingSelected(AdapterView<?> parent) {
            }
        });
    }
}
