package com.androidbook.triviaquiz12;

import java.util.Calendar;

import android.app.AlertDialog;
import android.app.DatePickerDialog;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.text.format.DateFormat;
import android.text.format.Time;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.DatePicker;
import android.widget.EditText;
import android.widget.Spinner;
import android.widget.TextView;

public class QuizSettingsActivity extends QuizActivity {
    SharedPreferences mGameSettings;
    static final int DATE_DIALOG_ID = 0;
    static final int PASSWORD_DIALOG_ID = 1;
    
    /** アクティビティが最初に生成されるときに呼び出される */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.settings);
        // 共有プレファレンスの読み込み
        mGameSettings = getSharedPreferences(GAME_PREFERENCES, Context.MODE_PRIVATE);
        // ニックネーム入力フィールドの初期化
        initNicknameEntry();
        // メールアドレス入力フィールドの初期化
        initEmailEntry();
        // パスワード入力ダイアログの初期化
        initPasswordChooser();
        // 生年月日入力用日付ピッカーの初期化
        initDatePicker();
        // 性別を選択するスピナーの初期化
        initGenderSpinner();
    }
    
    @Override
    protected void onDestroy() {
        Log.d(DEBUG_TAG, "SHARED PREFERENCES");
        Log.d(DEBUG_TAG, "ニックネーム: " + mGameSettings.getString(GAME_PREFERENCES_NICKNAME, "未設定"));
        Log.d(DEBUG_TAG, "メールアドレス: " + mGameSettings.getString(GAME_PREFERENCES_EMAIL, "未設定"));
        Log.d(DEBUG_TAG, "性別（男=1, 女=2, 秘密=0）: " + mGameSettings.getInt(GAME_PREFERENCES_GENDER, 0));
        Log.d(DEBUG_TAG, "パスワード: " + mGameSettings.getString(GAME_PREFERENCES_PASSWORD, "未設定"));
        Log.d(DEBUG_TAG, "生年月日: "
              + DateFormat.format("yyyy年 MMMM dd日", mGameSettings.getLong(GAME_PREFERENCES_DOB, 0)));
        super.onDestroy();
    }
    
    /**
     * ニックネーム入力フィールドの初期化
     */
    private void initNicknameEntry() {
        // ニックネームを保存
        final EditText nicknameText = (EditText) findViewById(R.id.EditText_Nickname);
        if (mGameSettings.contains(GAME_PREFERENCES_NICKNAME)) {
            nicknameText.setText(mGameSettings.getString(GAME_PREFERENCES_NICKNAME, ""));
        }
        nicknameText.setOnKeyListener(new View.OnKeyListener() {
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if ((event.getAction() == KeyEvent.ACTION_DOWN) && (keyCode == KeyEvent.KEYCODE_ENTER)) {
                    String strNickname = nicknameText.getText().toString();
                    Editor editor = mGameSettings.edit();
                    editor.putString(GAME_PREFERENCES_NICKNAME, strNickname);
                    editor.commit();
                    return true;
                }
                return false;
            }
        });
    }
    
    /**
     * メールアドレス入力フィールドの初期化
     */
    private void initEmailEntry() {
        // メールアドレスを保存
        final EditText emailText = (EditText) findViewById(R.id.EditText_Email);
        if (mGameSettings.contains(GAME_PREFERENCES_EMAIL)) {
            emailText.setText(mGameSettings.getString(GAME_PREFERENCES_EMAIL, ""));
        }
        emailText.setOnKeyListener(new View.OnKeyListener() {
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if ((event.getAction() == KeyEvent.ACTION_DOWN) && (keyCode == KeyEvent.KEYCODE_ENTER)) {
                    Editor editor = mGameSettings.edit();
                    editor.putString(GAME_PREFERENCES_EMAIL, emailText.getText().toString());
                    editor.commit();
                    return true;
                }
                return false;
            }
        });
    }
    
    /**
     * パスワード入力ダイアログの初期化
     */
    private void initPasswordChooser() {
        // パスワード設定状況の表示
        TextView passwordInfo = (TextView) findViewById(R.id.TextView_Password_Info);
        if (mGameSettings.contains(GAME_PREFERENCES_PASSWORD)) {
            passwordInfo.setText(R.string.settings_pwd_set);
        } else {
            passwordInfo.setText(R.string.settings_pwd_not_set);
        }
        // パスワード入力ダイアログの処理
        Button setPassword = (Button) findViewById(R.id.Button_Password);
        setPassword.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                showDialog(PASSWORD_DIALOG_ID);
            }
        });
    }
    
    /**
     * 生年月日入力用日付ピッカーの初期化
     */
    private void initDatePicker() {
        // 生年月日設定状況の表示
        TextView dobInfo = (TextView) findViewById(R.id.TextView_DOB_Info);
        if (mGameSettings.contains(GAME_PREFERENCES_DOB)) {
            dobInfo.setText(DateFormat.format("yyyy年 MMMM dd日", mGameSettings.getLong(GAME_PREFERENCES_DOB, 0)));
        } else {
            dobInfo.setText(R.string.settings_dob_not_set);
        }
        // 日付ピッカーダイアログの処理
        Button pickDate = (Button) findViewById(R.id.Button_DOB);
        pickDate.setOnClickListener(new View.OnClickListener() {
            public void onClick(View v) {
                showDialog(DATE_DIALOG_ID);
            }
        });
    }
    
    /**
     * 性別を選択するスピナーの初期化
     */
    private void initGenderSpinner() {
        // Spinnerコントロールに性別文字列を設定
        final Spinner spinner = (Spinner) findViewById(R.id.Spinner_Gender);
        ArrayAdapter<?> adapter = ArrayAdapter.createFromResource(this, R.array.genders,
                                                                  android.R.layout.simple_spinner_item);
        adapter.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        spinner.setAdapter(adapter);
        if (mGameSettings.contains(GAME_PREFERENCES_GENDER)) {
            spinner.setSelection(mGameSettings.getInt(GAME_PREFERENCES_GENDER, 0));
        }
        // スピナーによる選択の処理
        spinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            public void onItemSelected(AdapterView<?> parent, View itemSelected, int selectedItemPosition,
                                       long selectedId) {
                Editor editor = mGameSettings.edit();
                editor.putInt(GAME_PREFERENCES_GENDER, selectedItemPosition);
                editor.commit();
            }
            
            public void onNothingSelected(AdapterView<?> parent) {
            }
        });
    }
    
    @Override
    protected Dialog onCreateDialog(int id) {
        switch (id) {
            case DATE_DIALOG_ID:
                final TextView dob = (TextView) findViewById(R.id.TextView_DOB_Info);
                DatePickerDialog dateDialog = new DatePickerDialog(this, new DatePickerDialog.OnDateSetListener() {
                    public void onDateSet(DatePicker view, int year, int monthOfYear, int dayOfMonth) {
                        Time dateOfBirth = new Time();
                        dateOfBirth.set(dayOfMonth, monthOfYear, year);
                        long dtDob = dateOfBirth.toMillis(true);
                        dob.setText(DateFormat.format("yyyy年 MMMM dd日", dtDob));
                        Editor editor = mGameSettings.edit();
                        editor.putLong(GAME_PREFERENCES_DOB, dtDob);
                        editor.commit();
                    }
                }, 0, 0, 0);
                return dateDialog;
            case PASSWORD_DIALOG_ID:
                LayoutInflater inflater = (LayoutInflater) getSystemService(Context.LAYOUT_INFLATER_SERVICE);
                final View layout = inflater.inflate(R.layout.password_dialog, (ViewGroup) findViewById(R.id.root));
                final EditText p1 = (EditText) layout.findViewById(R.id.EditText_Pwd1);
                final EditText p2 = (EditText) layout.findViewById(R.id.EditText_Pwd2);
                final TextView error = (TextView) layout.findViewById(R.id.TextView_PwdProblem);
                p2.addTextChangedListener(new TextWatcher() {
                    @Override
                    public void afterTextChanged(Editable s) {
                        String strPass1 = p1.getText().toString();
                        String strPass2 = p2.getText().toString();
                        if (strPass1.equals(strPass2)) {
                            error.setText(R.string.settings_pwd_equal);
                        } else {
                            error.setText(R.string.settings_pwd_not_equal);
                        }
                    }
                    
                    // ... 他の必須オーバライドメソッドは何もしない
                    @Override
                    public void beforeTextChanged(CharSequence s, int start, int count, int after) {
                    }
                    
                    @Override
                    public void onTextChanged(CharSequence s, int start, int before, int count) {
                    }
                });
                AlertDialog.Builder builder = new AlertDialog.Builder(this);
                builder.setView(layout);
                // AlertDialogの設定
                builder.setTitle(R.string.settings_button_pwd);
                builder.setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int whichButton) {
                        // ダイアログは強制的に終了し、プールから削除するので
                        // 再使用はできない（キャッシュされていない）
                        QuizSettingsActivity.this.removeDialog(PASSWORD_DIALOG_ID);
                    }
                });
                builder.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int which) {
                        TextView passwordInfo = (TextView) findViewById(R.id.TextView_Password_Info);
                        String strPassword1 = p1.getText().toString();
                        String strPassword2 = p2.getText().toString();
                        if (strPassword1.equals(strPassword2)) {
                            Editor editor = mGameSettings.edit();
                            editor.putString(GAME_PREFERENCES_PASSWORD, strPassword1);
                            editor.commit();
                            passwordInfo.setText(R.string.settings_pwd_set);
                        } else {
                            Log.d(DEBUG_TAG, "パスワードが一致していません。保存せず、古いパスワードを維持します（設定されている場合）。");
                        }
                        // ダイアログは強制的に終了し、プールから削除するので
                        // 再使用はできない（キャッシュされていない）
                        QuizSettingsActivity.this.removeDialog(PASSWORD_DIALOG_ID);
                    }
                });
                // AlertDialogを作成し、戻り値として返す。
                AlertDialog passwordDialog = builder.create();
                return passwordDialog;
        }
        return null;
    }
    
    @Override
    protected void onPrepareDialog(int id, Dialog dialog) {
        super.onPrepareDialog(id, dialog);
        switch (id) {
            case DATE_DIALOG_ID:
                // DatePickerDialogの初期化処理はすべてここで行う
                DatePickerDialog dateDialog = (DatePickerDialog) dialog;
                int iDay,
                iMonth,
                iYear;
                // プレファレンス情報に生年月日データがあるか調べる
                if (mGameSettings.contains(GAME_PREFERENCES_DOB)) {
                    // プレファレンス情報から生年月日を取り出す
                    long msBirthDate = mGameSettings.getLong(GAME_PREFERENCES_DOB, 0);
                    Time dateOfBirth = new Time();
                    dateOfBirth.set(msBirthDate);
                    iDay = dateOfBirth.monthDay;
                    iMonth = dateOfBirth.month;
                    iYear = dateOfBirth.year;
                } else {
                    Calendar cal = Calendar.getInstance();
                    // 今日の日付を取り出す
                    iDay = cal.get(Calendar.DAY_OF_MONTH);
                    iMonth = cal.get(Calendar.MONTH);
                    iYear = cal.get(Calendar.YEAR);
                }
                // DatePickerの日付を生年月日、または
                // 現在日付に設定する
                dateDialog.updateDate(iYear, iMonth, iDay);
                return;
            case PASSWORD_DIALOG_ID:
                // パスワードダイアログの初期化があればすべてここで処理する。
                // このサンプルプログラムでは、一度表示したパスワードダイアログは再表示せず、
                // 新しく設定するので、ここでする作業はない。
                // 使い終わったパスワードダイアログは、removeDialog()を呼び出して
                // アクティビティのダイアログプールから削除している。
                // パスワードダイアログが必要になった場合は、生成し直す。
                return;
        }
    }
}
